/*
   Hyllian's Data Dependent Triangulation Shader w. Lottes curvature
   
   Copyright (C) 2011-2015 Hyllian - sergiogdb@gmail.com

   Permission is hereby granted, free of charge, to any person obtaining a copy
   of this software and associated documentation files (the "Software"), to deal
   in the Software without restriction, including without limitation the rights
   to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
   copies of the Software, and to permit persons to whom the Software is
   furnished to do so, subject to the following conditions:

   The above copyright notice and this permission notice shall be included in
   all copies or substantial portions of the Software.

   THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
   IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
   FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
   AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
   LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
   OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
   THE SOFTWARE.

*/


#define warpX 0.0375 // x-curvature, usable values between 0.0 - 0.1
#define warpY 0.0475 // y-curvature, usable values between 0.0 - 0.1


#include "shader.code"
string combineTechique : COMBINETECHNIQUE =  "DDT";

const static float3 dtt = float3(65536,255,1);

float reduce(float3 color)
{
	return dot(color, dtt);
}

float3 bilinear(float p, float q, float3 A, float3 B, float3 C, float3 D)
{
	return ((1-p)*(1-q)*A + p*(1-q)*B + (1-p)*q*C + p*q*D);
} 

// **VS**

VERTEX_STUFF1 DDT_VERTEX (float3 p : POSITION, float2 tc : TEXCOORD0)
{
  VERTEX_STUFF1 OUT = (VERTEX_STUFF1)0;
  
  float dx = ps.x*2.0;
  float dy = ps.y*2.0;

  OUT.coord = mul(float4(p,1),WorldViewProjection);
  OUT.CT = tc + float2(0.0000001, 0.0000001);
  OUT.UL = float2( dx, 0.0);
  OUT.UR = float2( 0.0, dy);
  return OUT;
}

// Distortion of scanlines, and end of screen alpha. (CRT-Lottes curvature)
float2 Warp(float2 pos)
{
   pos=pos*2.0-1.0;    
   pos*=float2(1.0+(pos.y*pos.y)*warpX,1.0+(pos.x*pos.x)*warpY);
   return pos*0.5+0.5;
}

// **PS**

float4 DDT_FRAGMENT ( in VERTEX_STUFF1 VAR ) : COLOR
{
	float2 coord=Warp(VAR.CT);

	float2 pos = frac(coord/(ps*2.0))-float2(0.5, 0.5); // pos = pixel position
	float2 dir = sign(pos); // dir = pixel direction

	float2 g1 = dir*VAR.UL.xy;
	float2 g2 = dir*VAR.UR.xy;

	float3 A = tex2D(s_p, coord       ).xyz;
	float3 B = tex2D(s_p, coord +g1   ).xyz;
	float3 C = tex2D(s_p, coord    +g2).xyz;
	float3 D = tex2D(s_p, coord +g1+g2).xyz;

	float a = reduce(A);
	float b = reduce(B);
	float c = reduce(C);
	float d = reduce(D);

	float p = abs(pos.x);
	float q = abs(pos.y);

	float k = distance(pos,g1);
	float l = distance(pos,g2);

	if (abs(a-d) < abs(b-c))
	{
		if (k < l)
		{
			C = A + D - B;
		}
		else if (k >= l)
		{
			B = A + D - C;
		}
	}
	else if (abs(a-d) > abs(b-c))
	{
		D = B + C - A;
	}

	float3 color = bilinear(p, q, A, B, C, D);

	return float4(color, 1.0); 
}

technique DDT
{
   pass P0
   {
     VertexShader = compile vs_3_0 DDT_VERTEX();
     PixelShader  = compile ps_3_0 DDT_FRAGMENT();
   }  
}
